<?php

declare(strict_types=1);

namespace App\Core;

final class JsonStorage
{
    private string $directory;

    public function __construct(string $directory)
    {
        $this->directory = rtrim($directory, DIRECTORY_SEPARATOR);
        if (!is_dir($this->directory)) {
            mkdir($this->directory, 0775, true);
        }
    }

    private function pathFor(string $collection): string
    {
        $safe = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $collection);
        return $this->directory . DIRECTORY_SEPARATOR . $safe . '.json';
    }

    public function load(string $collection): array
    {
        $path = $this->pathFor($collection);
        if (!is_file($path)) {
            return [];
        }
        $contents = file_get_contents($path);
        if ($contents === false || $contents === '') {
            return [];
        }
        $data = json_decode($contents, true);
        if (!is_array($data)) {
            return [];
        }
        return $data;
    }

    public function save(string $collection, array $data): void
    {
        $path = $this->pathFor($collection);
        $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        if ($json === false) {
            throw new \RuntimeException('Failed to encode JSON data.');
        }
        $tmpPath = $path . '.tmp';
        $bytes = file_put_contents($tmpPath, $json, LOCK_EX);
        if ($bytes === false) {
            throw new \RuntimeException('Failed to write JSON data.');
        }
        rename($tmpPath, $path);
    }

    public function update(string $collection, callable $callback): array
    {
        $data = $this->load($collection);
        $updated = $callback($data);
        if (!is_array($updated)) {
            throw new \RuntimeException('Update callback must return array.');
        }
        $this->save($collection, $updated);
        return $updated;
    }
}
