<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class BehaviorAnalysisModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return false;
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'behavior:') !== 0) {
            return;
        }

        $action = substr($data, strlen('behavior:'));
        if ($action === 'summary') {
            $this->handleSummary($update, $bot);
        }
    }

    private function handleSummary(array $update, TelegramBot $bot): void
    {
        $callback = $update['callback_query'] ?? [];
        $from = $callback['from'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($from['id'], $chat['id'])) {
            return;
        }

        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        $userId = (int) $from['id'];
        $isAdmin = in_array($userId, array_map('intval', $adminIds), true);
        if (!$isAdmin) {
            return;
        }

        $chatId = (int) $chat['id'];
        $storage = $bot->getStorage();

        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $result = $this->runAnalysis($bot, $mainGroupId);

        $groups = [
            'disciplined' => [],
            'consistent' => [],
            'passive' => [],
            'overactive' => [],
            'emotional' => [],
        ];

        foreach ($result as $row) {
            $label = (string) ($row['behavior_label'] ?? '');
            if (!isset($groups[$label])) {
                continue;
            }
            $groups[$label][] = $row;
        }

        $labelTitles = [
            'disciplined' => 'منظم و منضبط ✅',
            'consistent' => 'حضور پایدار 📈',
            'passive' => 'کم‌فعال 💤',
            'overactive' => 'بیش‌فعال / پرپیام 💬',
            'emotional' => 'رفتار هیجانی / نوسانی 🎢',
        ];

        $lines = [];
        $lines[] = '<b>🧠 تحلیل رفتاری هنرجویان (۱۴ روز اخیر)</b>';

        foreach ($groups as $label => $rows) {
            if (empty($rows)) {
                continue;
            }
            $lines[] = '';
            $lines[] = '🔹 <b>' . ($labelTitles[$label] ?? $label) . '</b>';
            foreach ($rows as $row) {
                $name = (string) ($row['name'] ?? '');
                $msg14 = (int) ($row['messages_14d'] ?? 0);
                $votes14 = (int) ($row['votes_14d'] ?? 0);
                $days = (int) ($row['active_days'] ?? 0);
                $ratio = (float) ($row['vote_message_ratio'] ?? 0.0);
                $ratioText = number_format($ratio, 2, '.', '');

                $lines[] = '• 👤 <b>' . $name . '</b>';
                $lines[] = '  📅 روزهای فعال: ' . $days . ' | ✉️ پیام‌ها (۱۴ روز): ' . $msg14 . ' | 🤝 رأی‌ها (۱۴ روز): ' . $votes14 . ' | ⚖️ نسبت رأی/پیام: ' . $ratioText;
            }
        }

        if (count($lines) === 1) {
            $lines[] = '';
            $lines[] = 'هنوز داده کافی برای تحلیل رفتاری وجود ندارد. بعد از چند روز فعالیت، این بخش پر می‌شود.';
        }

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '⬅️ بازگشت به منوی ادمین', 'callback_data' => 'admin:menu'],
                ],
            ],
        ];

        $text = implode("\n", $lines);

        if (isset($message['message_id'])) {
            $messageId = (int) $message['message_id'];
            $bot->getClient()->editMessageText($chatId, $messageId, $text, [
                'reply_markup' => $keyboard,
            ]);
        } else {
            $bot->getClient()->sendMessage($chatId, $text, [
                'reply_markup' => $keyboard,
            ]);
        }
    }

    private function runAnalysis(TelegramBot $bot, int $mainGroupId): array
    {
        $storage = $bot->getStorage();
        $topicActivity = $storage->load('topic_activity');
        $ratings = $storage->load('ratings');
        $users = $storage->load('users');
        $behaviorProfiles = $storage->load('behavior_profiles');
        $progressHistory = $storage->load('progress_history');
        $config = $bot->getConfig();
        $adminIds = $config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            $adminIds = [];
        }
        $adminIds = array_map('intval', $adminIds);

        if (!is_array($topicActivity)) {
            $topicActivity = [];
        }
        if (!is_array($ratings)) {
            $ratings = [];
        }
        if (!is_array($users)) {
            $users = [];
        }
        if (!is_array($behaviorProfiles)) {
            $behaviorProfiles = [];
        }
        if (!is_array($progressHistory)) {
            $progressHistory = [];
        }

        $now = new \DateTimeImmutable('now');
        $nowStr = $now->format('Y-m-d H:i:s');
        $windowStart = $now->modify('-13 days')->format('Y-m-d');

        $perUser = [];
        $perUserDays = [];

        if (isset($topicActivity[$mainGroupId]) && is_array($topicActivity[$mainGroupId])) {
            foreach ($topicActivity[$mainGroupId] as $topic) {
                if (!is_array($topic) || !isset($topic['users']) || !is_array($topic['users'])) {
                    continue;
                }

                foreach ($topic['users'] as $uid => $u) {
                    $uid = (int) $uid;
                    if (in_array($uid, $adminIds, true)) {
                        continue;
                    }
                    if (!isset($perUser[$uid])) {
                        $name = '';
                        if (isset($users[$uid]) && is_array($users[$uid])) {
                            $userRow = $users[$uid];
                            if (!empty($userRow['first_name'])) {
                                $name = (string) $userRow['first_name'];
                            } elseif (!empty($userRow['username'])) {
                                $name = '@' . (string) $userRow['username'];
                            }
                        }
                        if ($name === '') {
                            $name = 'کاربر ' . $uid;
                        }

                        $perUser[$uid] = [
                            'user_id' => $uid,
                            'name' => $name,
                            'total_messages' => 0,
                            'messages_14d' => 0,
                            'votes_14d' => 0,
                            'active_days' => 0,
                            'vote_message_ratio' => 0.0,
                            'ratings_sum' => 0,
                            'behavior_label' => 'passive',
                        ];
                        $perUserDays[$uid] = [];
                    }

                    $perUser[$uid]['total_messages'] += (int) ($u['message_count'] ?? 0);

                    $daily = $u['daily_messages'] ?? [];
                    if (is_array($daily)) {
                        foreach ($daily as $day => $count) {
                            if (!is_string($day) || $day === '') {
                                continue;
                            }
                            if ($day < $windowStart) {
                                continue;
                            }
                            $count = (int) $count;
                            if ($count <= 0) {
                                continue;
                            }
                            $perUser[$uid]['messages_14d'] += $count;
                            $perUserDays[$uid][$day] = true;
                        }
                    }
                }
            }
        }

        if (isset($ratings[$mainGroupId]) && is_array($ratings[$mainGroupId])) {
            foreach ($ratings[$mainGroupId] as $entry) {
                if (!is_array($entry)) {
                    continue;
                }
                $authorId = (int) ($entry['author_id'] ?? 0);
                if ($authorId !== 0) {
                    if (in_array($authorId, $adminIds, true)) {
                        continue;
                    }
                    if (!isset($perUser[$authorId])) {
                        $name = '';
                        if (isset($users[$authorId]) && is_array($users[$authorId])) {
                            $userRow = $users[$authorId];
                            if (!empty($userRow['first_name'])) {
                                $name = (string) $userRow['first_name'];
                            } elseif (!empty($userRow['username'])) {
                                $name = '@' . (string) $userRow['username'];
                            }
                        }
                        if ($name === '') {
                            $name = 'کاربر ' . $authorId;
                        }
                        $perUser[$authorId] = [
                            'user_id' => $authorId,
                            'name' => $name,
                            'total_messages' => 0,
                            'messages_14d' => 0,
                            'votes_14d' => 0,
                            'active_days' => 0,
                            'vote_message_ratio' => 0.0,
                            'ratings_sum' => 0,
                            'behavior_label' => 'passive',
                        ];
                        $perUserDays[$authorId] = [];
                    }
                    $statsEntry = $entry['stats'] ?? [];
                    $perUser[$authorId]['ratings_sum'] += (int) ($statsEntry['sum'] ?? 0);
                }

                if (isset($entry['votes']) && is_array($entry['votes'])) {
                    foreach ($entry['votes'] as $voterId => $vote) {
                        $voterId = (int) $voterId;
                        if (in_array($voterId, $adminIds, true)) {
                            continue;
                        }
                        $votedAt = (string) ($vote['voted_at'] ?? '');
                        if ($votedAt === '') {
                            continue;
                        }
                        try {
                            $dt = new \DateTimeImmutable($votedAt);
                        } catch (\Throwable $e) {
                            continue;
                        }
                        $day = $dt->format('Y-m-d');
                        if ($day < $windowStart) {
                            continue;
                        }

                        if (!isset($perUser[$voterId])) {
                            $name = '';
                            if (isset($users[$voterId]) && is_array($users[$voterId])) {
                                $userRow = $users[$voterId];
                                if (!empty($userRow['first_name'])) {
                                    $name = (string) $userRow['first_name'];
                                } elseif (!empty($userRow['username'])) {
                                    $name = '@' . (string) $userRow['username'];
                                }
                            }
                            if ($name === '') {
                                $name = 'کاربر ' . $voterId;
                            }

                            $perUser[$voterId] = [
                                'user_id' => $voterId,
                                'name' => $name,
                                'total_messages' => 0,
                                'messages_14d' => 0,
                                'votes_14d' => 0,
                                'active_days' => 0,
                                'vote_message_ratio' => 0.0,
                                'ratings_sum' => 0,
                                'behavior_label' => 'passive',
                            ];
                            $perUserDays[$voterId] = [];
                        }

                        $perUser[$voterId]['votes_14d']++;
                        $perUserDays[$voterId][$day] = true;
                    }
                }
            }
        }

        foreach ($perUser as $uid => &$row) {
            $days = isset($perUserDays[$uid]) ? count($perUserDays[$uid]) : 0;
            $row['active_days'] = $days;
            $msgs = (int) $row['messages_14d'];
            $votes = (int) $row['votes_14d'];
            if ($msgs > 0) {
                $row['vote_message_ratio'] = $votes / $msgs;
            } else {
                $row['vote_message_ratio'] = $votes > 0 ? (float) $votes : 0.0;
            }

            $label = 'passive';
            if ($msgs === 0 && $votes === 0) {
                $label = 'passive';
            } elseif ($days >= 8 && $msgs >= 10 && $msgs <= 80 && $row['vote_message_ratio'] >= 0.3 && $row['vote_message_ratio'] <= 1.5) {
                $label = 'disciplined';
            } elseif ($msgs > 80 && $row['vote_message_ratio'] < 0.2) {
                $label = 'overactive';
            } elseif ($msgs > 40 && $days <= 4) {
                $label = 'emotional';
            } elseif ($days >= 5) {
                $label = 'consistent';
            } else {
                $label = 'passive';
            }

            $row['behavior_label'] = $label;

            if (!isset($behaviorProfiles[$mainGroupId])) {
                $behaviorProfiles[$mainGroupId] = [];
            }
            $behaviorProfiles[$mainGroupId][$uid] = [
                'user_id' => $uid,
                'chat_id' => $mainGroupId,
                'calculated_at' => $nowStr,
                'messages_14d' => $row['messages_14d'],
                'votes_14d' => $row['votes_14d'],
                'active_days' => $row['active_days'],
                'vote_message_ratio' => $row['vote_message_ratio'],
                'behavior_label' => $label,
            ];

            if (!isset($progressHistory[$mainGroupId])) {
                $progressHistory[$mainGroupId] = [];
            }
            if (!isset($progressHistory[$mainGroupId][$uid]) || !is_array($progressHistory[$mainGroupId][$uid])) {
                $progressHistory[$mainGroupId][$uid] = [];
            }

            $historyList =& $progressHistory[$mainGroupId][$uid];
            $addSnapshot = true;
            if (!empty($historyList)) {
                $last = end($historyList);
                if (is_array($last) && isset($last['snapshot_at'])) {
                    try {
                        $lastDt = new \DateTimeImmutable((string) $last['snapshot_at']);
                        $diff = $now->diff($lastDt);
                        if ((int) $diff->days < 7) {
                            $addSnapshot = false;
                        }
                    } catch (\Throwable $e) {
                        $addSnapshot = true;
                    }
                }
                reset($historyList);
            }

            if ($addSnapshot) {
                $activityScore = $row['messages_14d'] + (2 * $row['votes_14d']) + (3 * $row['active_days']);
                $historyList[] = [
                    'snapshot_at' => $nowStr,
                    'total_messages' => $row['total_messages'],
                    'ratings_sum' => $row['ratings_sum'],
                    'messages_14d' => $row['messages_14d'],
                    'votes_14d' => $row['votes_14d'],
                    'active_days_14d' => $row['active_days'],
                    'activity_score' => $activityScore,
                    'behavior_label' => $label,
                ];
            }
        }
        unset($row);

        $storage->save('behavior_profiles', $behaviorProfiles);
        $storage->save('progress_history', $progressHistory);

        return array_values($perUser);
    }
}
