<?php

declare(strict_types=1);

namespace App\Modules;

use App\Bot\TelegramBot;

final class LeaderboardModule extends BaseModule
{
    public function supports(string $command): bool
    {
        return $command === '/leaderboard';
    }

    public function handle(string $command, array $update, TelegramBot $bot): void
    {
        $message = $update['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($chat['id'])) {
            return;
        }

        $chatId = (int) $chat['id'];

        $this->sendLeaderboard($bot, $chatId);
    }

    public function handleCallback(string $data, array $update, TelegramBot $bot): void
    {
        if (strpos($data, 'leaderboard:') !== 0) {
            return;
        }

        $action = substr($data, strlen('leaderboard:'));
        if ($action !== 'summary') {
            return;
        }

        $callback = $update['callback_query'] ?? [];
        $message = $callback['message'] ?? [];
        $chat = $message['chat'] ?? [];

        if (!isset($chat['id'])) {
            return;
        }

        $chatId = (int) $chat['id'];

        $this->sendLeaderboard($bot, $chatId);
    }

    private function sendLeaderboard(TelegramBot $bot, int $chatId): void
    {
        $storage = $bot->getStorage();
        $config = $bot->getConfig();

        $settings = $storage->load('settings');
        if (!isset($settings['main_group']['id'])) {
            $bot->getClient()->sendMessage($chatId, 'ابتدا باید یک گروه اصلی ثبت شود. در گروه ترید بنویس: "نصب ربات".');
            return;
        }

        $mainGroupId = (int) $settings['main_group']['id'];

        $topicActivity = $storage->load('topic_activity');
        $topicRoles = $storage->load('topic_roles');
        $ratings = $storage->load('ratings');
        $behaviorProfiles = $storage->load('behavior_profiles');
        $users = $storage->load('users');

        if (!is_array($topicActivity)) {
            $topicActivity = [];
        }
        if (!is_array($topicRoles)) {
            $topicRoles = [];
        }
        if (!is_array($ratings)) {
            $ratings = [];
        }
        if (!is_array($behaviorProfiles)) {
            $behaviorProfiles = [];
        }
        if (!is_array($users)) {
            $users = [];
        }

        $adminIds = $config['admin_ids'] ?? [];
        if (!is_array($adminIds)) {
            $adminIds = [];
        }
        $adminIds = array_map('intval', $adminIds);

        $rolesForChat = $topicRoles[$mainGroupId] ?? [];
        $tradingTopicId = isset($rolesForChat['trading']['topic_id']) ? (int) $rolesForChat['trading']['topic_id'] : null;
        $warTopicId = isset($rolesForChat['war']['topic_id']) ? (int) $rolesForChat['war']['topic_id'] : null;
        $qaTopicId = isset($rolesForChat['qa']['topic_id']) ? (int) $rolesForChat['qa']['topic_id'] : null;

        $perUser = [];

        if (isset($topicActivity[$mainGroupId]) && is_array($topicActivity[$mainGroupId])) {
            foreach ($topicActivity[$mainGroupId] as $topicId => $topic) {
                if (!is_array($topic) || !isset($topic['users']) || !is_array($topic['users'])) {
                    continue;
                }

                $isTrading = $tradingTopicId !== null && (int) $topicId === $tradingTopicId;

                foreach ($topic['users'] as $uid => $u) {
                    $uid = (int) $uid;
                    if (in_array($uid, $adminIds, true)) {
                        continue;
                    }

                    if (!isset($perUser[$uid])) {
                        $perUser[$uid] = [
                            'user_id' => $uid,
                            'messages_trading' => 0,
                            'messages_war' => 0,
                            'messages_qa' => 0,
                            'messages_total' => 0,
                            'ratings_sum' => 0,
                            'rated_analyses' => 0,
                            'votes_cast' => 0,
                            'score' => 0,
                        ];
                    }

                    $count = (int) ($u['message_count'] ?? 0);
                    $perUser[$uid]['messages_total'] += $count;

                    $isTrading = $tradingTopicId !== null && (int) $topicId === $tradingTopicId;
                    $isWar = $warTopicId !== null && (int) $topicId === $warTopicId;
                    $isQa = $qaTopicId !== null && (int) $topicId === $qaTopicId;

                    if ($isTrading) {
                        $perUser[$uid]['messages_trading'] += $count;
                    } elseif ($isWar) {
                        $perUser[$uid]['messages_war'] += $count;
                    } elseif ($isQa) {
                        $perUser[$uid]['messages_qa'] += $count;
                    }
                }
            }
        }

        if (isset($ratings[$mainGroupId]) && is_array($ratings[$mainGroupId])) {
            foreach ($ratings[$mainGroupId] as $entry) {
                if (!is_array($entry)) {
                    continue;
                }
                $authorId = isset($entry['author_id']) ? (int) $entry['author_id'] : 0;
                if ($authorId !== 0 && !in_array($authorId, $adminIds, true)) {
                    if (!isset($perUser[$authorId])) {
                        $perUser[$authorId] = [
                            'user_id' => $authorId,
                            'messages_trading' => 0,
                            'messages_war' => 0,
                            'messages_qa' => 0,
                            'messages_total' => 0,
                            'ratings_sum' => 0,
                            'rated_analyses' => 0,
                            'votes_cast' => 0,
                            'score' => 0,
                        ];
                    }
                    $statsEntry = $entry['stats'] ?? [];
                    $perUser[$authorId]['ratings_sum'] += (int) ($statsEntry['sum'] ?? 0);
                    $perUser[$authorId]['rated_analyses']++;
                }

                if (isset($entry['votes']) && is_array($entry['votes'])) {
                    foreach ($entry['votes'] as $voterId => $vote) {
                        $voterId = (int) $voterId;
                        if (in_array($voterId, $adminIds, true)) {
                            continue;
                        }
                        if (!isset($perUser[$voterId])) {
                            $perUser[$voterId] = [
                                'user_id' => $voterId,
                                'messages_trading' => 0,
                                'messages_war' => 0,
                                'messages_qa' => 0,
                                'messages_total' => 0,
                                'ratings_sum' => 0,
                                'rated_analyses' => 0,
                                'votes_cast' => 0,
                                'score' => 0,
                            ];
                        }
                        $perUser[$voterId]['votes_cast']++;
                    }
                }
            }
        }

        if (empty($perUser)) {
            $bot->getClient()->sendMessage($chatId, 'هنوز داده‌ای برای ساخت لیدربورد هنرجویان در گروه اصلی وجود ندارد.');
            return;
        }

        foreach ($perUser as $uid => &$row) {
            $uid = (int) $uid;
            $messagesTrading = (int) $row['messages_trading'];
            $messagesWar = (int) ($row['messages_war'] ?? 0);
            $messagesQa = (int) ($row['messages_qa'] ?? 0);
            $ratedAnalyses = (int) ($row['rated_analyses'] ?? 0);
            $ratingsSum = (int) $row['ratings_sum'];
            $votesCast = (int) $row['votes_cast'];

            $scoreTradingRaw = ($messagesTrading * 1.0)
                + ($ratedAnalyses * 3.0)
                + ($ratingsSum * 2.0);
            // هر ۱۰ پیام در اقتصاد/جنگ و پرسش‌وپاسخ = ۱ امتیاز
            $scoreWarRaw = (float) floor($messagesWar / 10);
            $scoreQaRaw = (float) floor($messagesQa / 10);
            $scoreVotesRaw = $votesCast * 0.5;

            $totalScore = (int) round($scoreTradingRaw + $scoreWarRaw + $scoreQaRaw + $scoreVotesRaw);
            if ($totalScore < 0) {
                $totalScore = 0;
            }

            $row['score'] = $totalScore;
        }
        unset($row);

        $list = array_values($perUser);
        usort($list, function (array $a, array $b): int {
            $sa = (int) ($a['score'] ?? 0);
            $sb = (int) ($b['score'] ?? 0);
            if ($sa === $sb) {
                $ra = (int) ($a['ratings_sum'] ?? 0);
                $rb = (int) ($b['ratings_sum'] ?? 0);
                if ($ra === $rb) {
                    $ma = (int) ($a['messages_trading'] ?? 0);
                    $mb = (int) ($b['messages_trading'] ?? 0);
                    return $mb <=> $ma;
                }
                return $rb <=> $ra;
            }
            return $sb <=> $sa;
        });

        $top = array_slice($list, 0, 10);

        $lines = [];
        $lines[] = '<b>🏆 لیدربورد تریدرهای برتر</b>';
        $lines[] = '🔥 بر اساس امتیاز فعالیت در ترید هنرجویان، کیفیت تحلیل‌ها و مشارکت.';

        $rank = 1;
        foreach ($top as $row) {
            $uid = (int) ($row['user_id'] ?? 0);
            $userRow = $users[$uid] ?? [];
            $name = '';
            if (is_array($userRow) && isset($userRow['first_name']) && $userRow['first_name'] !== '') {
                $name = (string) $userRow['first_name'];
            } elseif (is_array($userRow) && isset($userRow['username']) && $userRow['username'] !== '') {
                $name = '@' . (string) $userRow['username'];
            } else {
                $name = 'کاربر ' . $uid;
            }

            $score = (int) ($row['score'] ?? 0);

            if ($score >= 900) {
                $levelLabel = 'سطح ۵ - اسطوره مارکت 👑';
            } elseif ($score >= 400) {
                $levelLabel = 'سطح ۴ - استراتژیست حرفه‌ای 🧠';
            } elseif ($score >= 250) {
                $levelLabel = 'سطح ۳ - تریدر پایدار 🔥';
            } elseif ($score >= 200) {
                $levelLabel = 'سطح ۲ - فعال رو به رشد 🚀';
            } else {
                $levelLabel = 'سطح ۱ - تازه‌کار بازار 🐣';
            }

            $medal = '🏅';
            if ($rank === 1) {
                $medal = '🥇';
            } elseif ($rank === 2) {
                $medal = '🥈';
            } elseif ($rank === 3) {
                $medal = '🥉';
            }

            $line = $medal . ' <b>' . $name . '</b>' . "\n";
            $line .= 'سطح: ' . $levelLabel . ' | امتیاز لیدربورد: ' . $score;

            $lines[] = $line;

            $rank++;
        }

        $bot->getClient()->sendMessage($chatId, implode("\n\n", $lines));
    }
}
